/*
 * Data structures for message tunnels
 */

#include "fma_myri_packet.h"

#define FMA_TUNNEL_ACK_TIMEOUT 1000
#define FMA_TUNNEL_NUM_RETRIES 20
#define FMA_TUNNEL_RX_TIMEOUT (FMA_TUNNEL_ACK_TIMEOUT*FMA_TUNNEL_NUM_RETRIES)

/*
 * Descriptor for transmit tunnel instance
 */
struct fma_tunnel_tx_desc {
  void *msg1;			/* start of 1st buffer */
  int length1;			/* 1st length */
  void *msg2;			/* start of 2nd buffer */
  int length2;			/* 2nd length */
  int msg_id;			/* unique ID */

  uint8_t route[MYRI_MAX_ROUTE_LEN];	/* route to dest */
  int route_len;
  int nic_handle;			/* NIC to use */
  int port;
  lf_mac_addr_t my_mac_addr;		/* mac addr of this NIC */

  int offset_to_send;		/* next offset to send */

  /* allocated dynamically since size if variable */
  struct fma_myri_packet *pkt_ptr;	/* packet we are sending */
  int last_pkt_len;			/* length of last packet sent */

  void *context;		/* user pointer for callbacks */
  void (*complete_rtn)(void *);	/* successful completion routine */
  void (*error_rtn)(void *);	/* error routine */

  struct lf_event *ack_timer;	/* ACK wait timer */
  int retries_left;

  int sends_pending;		/* # of sends in queue */
  int stale;			/* stale means this is complete, but sends
				   are still pending */

  struct fma_tunnel_tx_desc *next;	/* links for active list */
  struct fma_tunnel_tx_desc *prev;
};

/*
 * Descriptor for receive tunnel instance
 */
struct fma_tunnel_rx_desc {
  int type;
  int length;			/* total length */
  void *msg;			/* start of message */

  int msg_id;			/* unique ID */
  lf_mac_addr_t sender_mac_addr; /* mac addr of the sender */

  uint8_t reply_route[MYRI_MAX_ROUTE_LEN];	/* route to dest */
  int reply_route_len;

  struct fma_nic_info *nip;	/* NIC/port to use for reply */
  int port;

  int next_offset;		/* next offset to expect */

  /* static since it is just an ACK */
  struct fma_myri_packet pkt;	/* packet we are sending */

  struct lf_event *msg_timer;	/* Inter-packet timer */

  struct fma_tunnel_rx_desc *next;	/* links for active list */
  struct fma_tunnel_rx_desc *prev;
};

/*
 * Global struct for tunnel data
 */
struct fma_tunnel {
  int tunnel_msg_id;
  struct fma_tunnel_tx_desc tx_anchor; /* list anchor of pending tunnel xmits */
  struct fma_tunnel_rx_desc rx_anchor; /* list anchor of pending tunnel recvs */
};

/*
 * Tunnel prototypes
 */
int fma_tunnel_send(void *msg1, int length1, void *msg2, int length2,
  uint8_t *route, int route_len, struct fma_nic_info *nip, int port,
  void (*complete_rtn)(void *), void (*error_rtn)(void *), void *context);
void fma_tunnel_cancel_send(int msg_id);
void fma_tunnel_got_ack(struct fma_myri_packet *pkt);
void fma_tunnel_got_start(struct fma_nic_info *nip, int port,
                          struct fma_myri_packet *pkt);
void fma_tunnel_got_body(struct fma_myri_packet *pkt);
